<?php
require_once 'includes/auth.php';

$msg = '';
$msgType = 'success';

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $act = $_POST['action'] ?? '';

    try {
        if ($act === 'approve_agent_commission') {
            $reqId = (int)$_POST['request_id'];

            $db->beginTransaction();

            $sel = $db->prepare("SELECT * FROM agent_commission_requests WHERE id = ? AND status = 'pending' FOR UPDATE");
            $sel->execute([$reqId]);
            $req = $sel->fetch();

            if (!$req) throw new Exception('Request not found or already processed.');

            // Mark approved
            $db->prepare("UPDATE agent_commission_requests SET status='approved', admin_note=?, updated_at=NOW() WHERE id=?")
               ->execute(['Approved by admin', $reqId]);

            // Log to referral_payouts so available commission decreases
            $db->prepare("INSERT INTO referral_payouts (referrer_id, amount, created_at) VALUES (?, ?, NOW())")
               ->execute([$req['agent_id'], $req['amount']]);

            $db->commit();
            $msg = "✅ Payout request #{$reqId} approved. ₹" . money($req['amount']) . " marked as paid to agent " . h($req['agent_id']);
            $msgType = 'success';
        }

        if ($act === 'reject_agent_commission') {
            $reqId  = (int)$_POST['request_id'];
            $note   = trim($_POST['admin_note'] ?? 'Rejected by admin');

            $db->prepare("UPDATE agent_commission_requests SET status='rejected', admin_note=?, updated_at=NOW() WHERE id=? AND status='pending'")
               ->execute([$note, $reqId]);

            $msg = "Request #{$reqId} rejected. Reason: " . h($note); 
            $msgType = 'success';
        }
    } catch (Exception $e) {
        if ($db->inTransaction()) $db->rollBack();
        $msg = "Error: " . $e->getMessage();
        $msgType = 'error';
    }
}

// Fetch Filters
$filterStatus = $_GET['status'] ?? 'pending';
$allowedStatuses = ['pending', 'approved', 'rejected', 'all'];
if (!in_array($filterStatus, $allowedStatuses)) $filterStatus = 'pending';

// Fetch Requests
$sqlFields = "SELECT * FROM agent_commission_requests";
if ($filterStatus !== 'all') {
    $sqlFields .= " WHERE status = " . $db->quote($filterStatus);
}
$sqlFields .= " ORDER BY id DESC LIMIT 100";
$agentRequests = $db->query($sqlFields)->fetchAll();

require_once 'includes/header.php';
?>

<div class="mb-8 flex flex-col md:flex-row md:items-end justify-between gap-4">
    <div>
        <h1 class="text-2xl font-bold text-white tracking-tight">Agent Payouts</h1>
        <p class="text-sm text-slate-400 mt-1">Review and process commission withdrawal requests.</p>
    </div>
    
    <div class="flex items-center gap-1.5 bg-slate-800/80 p-1 rounded-xl border border-slate-700/50 flex-wrap">
        <?php foreach(['pending'=>'⏳ Pending', 'approved'=>'✅ Approved', 'rejected'=>'❌ Rejected', 'all'=>'All'] as $s => $label): ?>
            <a href="?status=<?= $s ?>" class="px-4 py-2 rounded-lg text-sm font-bold transition-all <?= $filterStatus === $s ? 'bg-blue-600 text-white shadow-lg shadow-blue-600/20' : 'text-slate-400 hover:text-white hover:bg-slate-700' ?>">
                <?= $label ?>
            </a>
        <?php endforeach; ?>
    </div>
</div>

<?php if($msg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border <?= $msgType === 'success' ? 'bg-green-500/10 border-green-500/20 text-green-400' : 'bg-red-500/10 border-red-500/20 text-red-400' ?>">
        <i data-feather="<?= $msgType === 'success' ? 'check-circle' : 'alert-circle' ?>" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($msg) ?></p>
    </div>
<?php endif; ?>

<div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden">
    <?php if(empty($agentRequests)): ?>
        <div class="p-12 text-center text-slate-500 font-medium flex flex-col items-center">
            <div class="w-16 h-16 bg-slate-800 rounded-full flex items-center justify-center mb-4">
                <i data-feather="inbox" class="w-8 h-8 text-slate-600"></i>
            </div>
            No <?= h($filterStatus === 'all' ? '' : $filterStatus) ?> agent payout requests found.
        </div>
    <?php else: ?>
        <div class="overflow-x-auto">
            <table class="w-full text-left border-collapse">
                <thead>
                    <tr class="bg-slate-900/50 border-b border-slate-700/50">
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest whitespace-nowrap">Request ID</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest whitespace-nowrap">Agent ID</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest whitespace-nowrap">Claim Amount</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest min-w-[200px]">Bank / Crypto Details</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Status / Date</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-right min-w-[120px]">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-slate-800/50">
                    <?php foreach ($agentRequests as $req):
                        $details = json_decode($req['payout_details'], true) ?? [];
                        $statusColors = [
                            'pending' => 'bg-yellow-500/10 text-yellow-500 border-yellow-500/20',
                            'approved' => 'bg-green-500/10 text-green-400 border-green-500/20',
                            'rejected' => 'bg-red-500/10 text-red-400 border-red-500/20'
                        ];
                        $statusColor = $statusColors[$req['status']] ?? 'bg-blue-500/10 text-blue-400';
                    ?>
                    <tr class="hover:bg-slate-800/40 transition-colors group">
                        <td class="p-4 text-sm font-bold text-slate-500">
                            #<?= $req['id'] ?>
                        </td>
                        <td class="p-4">
                            <span class="text-sm font-bold text-indigo-400 flex items-center gap-1.5"><i data-feather="briefcase" class="w-4 h-4 text-indigo-500 mt-0.5"></i> <?= h($req['agent_id']) ?></span>
                        </td>
                        <td class="p-4">
                            <span class="text-xl font-black text-white tracking-tight">₹<?= money($req['amount']) ?></span>
                        </td>
                        <td class="p-4">
                            <div class="mb-2">
                                <span class="bg-slate-700 text-slate-300 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider"><?= h($req['payout_method']) ?></span>
                            </div>
                            <div class="space-y-0.5">
                                <?php foreach ($details as $k => $v): ?>
                                    <div class="text-[11px] font-mono"><span class="text-slate-500"><?= h($k) ?>:</span> <span class="text-slate-300"><?= h($v) ?></span></div>
                                <?php endforeach; ?>
                                <?php if ($req['admin_note']): ?>
                                    <div class="text-[11px] font-bold text-orange-400 italic mt-1.5 border-l-2 border-orange-500/50 pl-2">Admin Note: <?= h($req['admin_note']) ?></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td class="p-4">
                            <span class="px-2.5 py-1 rounded text-[10px] font-bold uppercase tracking-wider border <?= $statusColor ?>">
                                <?= h($req['status']) ?>
                            </span>
                            <div class="text-[10px] text-slate-500 mt-2 font-medium flex items-center gap-1"><i data-feather="clock" class="w-3 h-3"></i> <?= date('d M y H:i', strtotime($req['created_at'])) ?></div>
                        </td>
                        <td class="p-4 text-right">
                            <?php if ($req['status'] === 'pending'): ?>
                                <div class="flex flex-col gap-2">
                                    <form method="POST" action="?status=<?= $filterStatus ?>" onsubmit="return confirm('Confirm approval of ₹<?= money($req['amount']) ?> for agent <?= h($req['agent_id']) ?>?\n\nThis marks the commission as PAID. Make sure you have manually transferred the money first.');">
                                        <input type="hidden" name="action" value="approve_agent_commission">
                                        <input type="hidden" name="request_id" value="<?= $req['id'] ?>">
                                        <button type="submit" class="w-full bg-green-500/20 hover:bg-green-500/30 text-green-400 font-bold border border-green-500/30 px-3 py-1.5 rounded-lg text-xs transition-colors flex justify-center items-center gap-1.5"><i data-feather="check" class="w-3 h-3"></i> Approve</button>
                                    </form>
                                    <form method="POST" action="?status=<?= $filterStatus ?>" onsubmit="var note=prompt('Rejection reason:',''); if(!note) return false; this.querySelector('[name=admin_note]').value=note; return true;">
                                        <input type="hidden" name="action" value="reject_agent_commission">
                                        <input type="hidden" name="request_id" value="<?= $req['id'] ?>">
                                        <input type="hidden" name="admin_note" value="">
                                        <button type="submit" class="w-full bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold border border-red-500/20 px-3 py-1.5 rounded-lg text-xs transition-colors flex justify-center items-center gap-1.5"><i data-feather="x" class="w-3 h-3"></i> Reject</button>
                                    </form>
                                </div>
                            <?php else: ?>
                                <span class="text-slate-600 text-sm font-bold">— Processing Complete —</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php require_once 'includes/footer.php'; ?>
