<?php
require_once 'includes/auth.php';

$searchId = trim($_GET['search_user'] ?? '');
$userData = null;
$errorMsg = '';

if ($searchId) {
    // 1. Get User Details
    $uStmt = $db->prepare("SELECT * FROM users WHERE user_id = ?");
    $uStmt->execute([$searchId]);
    $userData = $uStmt->fetch();

    if ($userData) {
        // 2. Get Total Deposit
        $depStmt = $db->prepare("SELECT SUM(amount) FROM transactions WHERE user_id = ? AND type = 'deposit' AND status = 'completed'");
        $depStmt->execute([$searchId]);
        $totalDeposit = $depStmt->fetchColumn() ?: 0;

        // 3. Get Total Withdrawal
        $wdStmt = $db->prepare("SELECT SUM(amount) FROM transactions WHERE user_id = ? AND type = 'withdrawal' AND status = 'completed'");
        $wdStmt->execute([$searchId]);
        $totalWithdraw = $wdStmt->fetchColumn() ?: 0;
        
        // 4. Get Network Size
        $refStmt = $db->prepare("SELECT COUNT(*) FROM users WHERE referred_by = ?");
        $refStmt->execute([$searchId]);
        $networkSize = $refStmt->fetchColumn() ?: 0;

        // Fetch Recent Transactions
        $txStmt = $db->prepare("SELECT type, amount, status, created_at FROM transactions WHERE user_id = ? AND type IN ('deposit', 'withdrawal', 'admin_deposit', 'admin_withdraw') ORDER BY id DESC LIMIT 50");
        $txStmt->execute([$searchId]);
        $txs = $txStmt->fetchAll();

        // Fetch Recent Game Bets (Union of Wingo, Mines, Crash)
        $bets = [];
        try {
            $w = $db->prepare("SELECT 'Wingo' as game, amount as bet, win_amount as win, created_at FROM wingo_bets WHERE user_id = ? ORDER BY id DESC LIMIT 20");
            $w->execute([$searchId]);
            foreach($w as $r) $bets[] = $r;
        } catch (\PDOException $e) {}
            
        try {
            $c = $db->prepare("SELECT 'Crash' as game, amount as bet, win_amount as win, created_at FROM crash_bets WHERE user_id = ? ORDER BY id DESC LIMIT 20");
            $c->execute([$searchId]);
            foreach($c as $r) $bets[] = $r;
        } catch (\PDOException $e) {}

        try {
            $m = $db->prepare("SELECT 'Mines' as game, bet_amount as bet, COALESCE(total_payout, 0) as win, created_at FROM bets WHERE user_id = ? ORDER BY id DESC LIMIT 20");
            $m->execute([$searchId]);
            foreach($m as $r) $bets[] = $r;
        } catch (\PDOException $e) {}

        usort($bets, function($a, $b) {
            return strtotime($b['created_at']) <=> strtotime($a['created_at']);
        });
        $bets = array_slice($bets, 0, 50); // Keep top 50 recent overall
    } else {
        $errorMsg = "User ID '$searchId' not found in the database.";
    }
}

require_once 'includes/header.php';
?>

<div class="mb-8 flex flex-col md:flex-row md:items-end justify-between gap-4">
    <div>
        <h1 class="text-2xl font-bold text-white tracking-tight flex items-center gap-2">
            <i data-feather="globe" class="w-6 h-6 text-indigo-500"></i> Globe Lens
        </h1>
        <p class="text-sm text-slate-400 mt-1">Deep dive into a specific player's entire unified history.</p>
    </div>
    
    <form method="GET" class="flex items-center gap-2">
        <div class="relative w-full md:w-64">
            <i data-feather="search" class="w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 text-slate-500"></i>
            <input type="text" name="search_user" value="<?= h($searchId) ?>" placeholder="Target User ID" class="w-full bg-slate-800/80 border border-slate-700/50 rounded-xl pl-9 pr-4 py-2.5 text-sm text-white placeholder:text-slate-500 focus:outline-none focus:ring-2 focus:ring-indigo-500/50 font-mono tracking-wider font-bold" required>
        </div>
        <button type="submit" class="bg-indigo-600 hover:bg-indigo-500 text-white px-4 py-2.5 rounded-xl font-bold text-sm transition-colors border border-indigo-500/50 shadow-lg shadow-indigo-600/20">Analyze</button>
        <?php if($searchId): ?>
            <a href="globe.php" class="bg-slate-800 hover:bg-slate-700 text-slate-300 px-4 py-2.5 rounded-xl font-bold text-sm transition-colors border border-slate-700">Clear</a>
        <?php endif; ?>
    </form>
</div>

<?php if($errorMsg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border bg-red-500/10 border-red-500/20 text-red-400">
        <i data-feather="x-circle" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($errorMsg) ?></p>
    </div>
<?php endif; ?>

<?php if(!$searchId && !$errorMsg): ?>
    <div class="border border-slate-700/50 bg-slate-800/20 rounded-2xl border-dashed p-16 flex flex-col items-center justify-center text-center">
        <div class="w-20 h-20 bg-slate-800 rounded-full flex items-center justify-center mb-6">
            <i data-feather="aperture" class="w-10 h-10 text-slate-600"></i>
        </div>
        <h3 class="text-xl font-bold text-white mb-2">Awaiting Target</h3>
        <p class="text-slate-400 max-w-sm">Enter a User ID in the search bar above to pull their complete financial and betting timeline.</p>
    </div>
<?php endif; ?>

<?php if($userData): ?>
    <!-- Top Identity Card -->
    <div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 p-6 mb-8 relative overflow-hidden">
        <div class="absolute -right-12 -top-12 w-48 h-48 bg-indigo-500/10 rounded-full blur-3xl pointer-events-none"></div>
        
        <div class="flex flex-col lg:flex-row justify-between lg:items-center gap-6 relative z-10">
            <div class="flex items-center gap-5">
                <div class="w-16 h-16 rounded-2xl bg-slate-700 flex items-center justify-center flex-shrink-0 border <?= $userData['is_banned'] ? 'border-red-500 text-red-500' : 'border-slate-600 text-slate-400' ?>">
                    <i data-feather="<?= $userData['is_banned'] ? 'slash' : 'user' ?>" class="w-8 h-8"></i>
                </div>
                <div>
                    <div class="flex items-center gap-3 mb-1">
                        <h2 class="text-3xl font-black text-white"><?= h($userData['user_id']) ?></h2>
                        <?php if($userData['is_banned']): ?>
                            <span class="bg-red-500/20 text-red-400 border border-red-500/30 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">Banned</span>
                        <?php else: ?>
                            <span class="bg-green-500/20 text-green-400 border border-green-500/30 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">Active</span>
                        <?php endif; ?>
                        
                        <?php if($userData['is_tutor']): ?>
                            <span class="bg-purple-500/20 text-purple-400 border border-purple-500/30 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">Tutor</span>
                        <?php endif; ?>
                    </div>
                    <p class="text-slate-400 font-medium">Contact: <span class="text-white font-bold"><?= h($userData['phone']) ?></span></p>
                    <p class="text-slate-500 text-xs mt-0.5">Joined: <?= date('F j, Y, g:i a', strtotime($userData['created_at'])) ?></p>
                </div>
            </div>

            <div class="flex flex-col gap-1 items-end">
                <p class="text-xs font-bold text-slate-500 uppercase tracking-widest text-right">Current Wallet Balance</p>
                <div class="text-4xl font-black text-green-400 tracking-tight text-right">₹<?= money($userData['balance']) ?></div>
            </div>
        </div>
    </div>

    <!-- Stats Grid -->
    <div class="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
        <div class="bg-slate-800/50 border border-slate-700/50 rounded-2xl p-5">
            <div class="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1 flex items-center gap-1.5"><i data-feather="arrow-down-left" class="w-3 h-3 text-emerald-400"></i> Total Deposited</div>
            <div class="text-xl font-black text-emerald-400">₹<?= money($totalDeposit) ?></div>
        </div>
        <div class="bg-slate-800/50 border border-slate-700/50 rounded-2xl p-5">
            <div class="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1 flex items-center gap-1.5"><i data-feather="arrow-up-right" class="w-3 h-3 text-red-400"></i> Total Withdrawn</div>
            <div class="text-xl font-black text-red-400">₹<?= money($totalWithdraw) ?></div>
        </div>
        <div class="bg-slate-800/50 border border-slate-700/50 rounded-2xl p-5">
            <div class="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1 flex items-center gap-1.5"><i data-feather="user-plus" class="w-3 h-3 text-indigo-400"></i> Referred By</div>
            <div class="text-xl font-black text-indigo-400"><?= !empty($userData['referred_by']) ? h($userData['referred_by']) : '<span class="text-slate-600">Organic</span>' ?></div>
        </div>
        <div class="bg-slate-800/50 border border-slate-700/50 rounded-2xl p-5">
            <div class="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1 flex items-center gap-1.5"><i data-feather="users" class="w-3 h-3 text-blue-400"></i> Direct Referrals</div>
            <div class="text-xl font-black text-blue-400"><?= number_format($networkSize) ?></div>
        </div>
    </div>

    <!-- Timelines -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        
        <!-- Money Movement -->
        <div class="border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden flex flex-col">
            <div class="p-5 border-b border-slate-700/50 flex items-center justify-between bg-slate-800/80">
                <h2 class="text-base font-bold text-white flex items-center gap-2">
                    <i data-feather="dollar-sign" class="w-4 h-4 text-emerald-400"></i> Financial History
                </h2>
            </div>
            <div class="overflow-y-auto max-h-[500px]">
                <?php if(empty($txs)): ?>
                    <div class="p-8 text-center text-slate-500 font-medium text-sm">No transactions on record.</div>
                <?php else: ?>
                    <table class="w-full text-left border-collapse">
                        <thead class="sticky top-0 bg-slate-900 z-10">
                            <tr class="border-b border-slate-700/50">
                                <th class="p-3 pl-5 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Date / Time</th>
                                <th class="p-3 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Type</th>
                                <th class="p-3 text-[10px] font-bold text-slate-400 uppercase tracking-widest right-align">Amount</th>
                                <th class="p-3 pr-5 text-[10px] font-bold text-slate-400 uppercase tracking-widest text-right">Status</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-slate-800/50">
                            <?php foreach($txs as $tx): 
                                $isDep = (strpos($tx['type'], 'deposit') !== false);
                            ?>
                            <tr class="hover:bg-slate-800/40 transition-colors">
                                <td class="p-3 pl-5 text-xs text-slate-500 font-mono"><?= date('M d H:i', strtotime($tx['created_at'])) ?></td>
                                <td class="p-3">
                                    <span class="text-xs font-bold uppercase tracking-wider <?= $isDep ? 'text-green-400' : 'text-red-400' ?>"><?= $isDep ? 'Dep' : 'W/D' ?></span>
                                </td>
                                <td class="p-3 font-bold text-white tracking-tight">₹<?= money($tx['amount']) ?></td>
                                <td class="p-3 pr-5 text-right">
                                    <?php 
                                        $sColor = 'bg-yellow-500 text-yellow-500 border-yellow-500';
                                        if($tx['status']==='completed') $sColor = 'bg-green-500 text-green-400 border-green-500';
                                        elseif($tx['status']==='rejected' || $tx['status']==='failed') $sColor = 'bg-red-500 text-red-400 border-red-500';
                                    ?>
                                    <span class="inline-block px-2 py-0.5 rounded text-[9px] font-bold uppercase tracking-wider bg-opacity-10 border border-opacity-20 <?= $sColor ?>">
                                        <?= h($tx['status']) ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

        <!-- Gameplay History -->
        <div class="border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden flex flex-col">
            <div class="p-5 border-b border-slate-700/50 flex items-center justify-between bg-slate-800/80">
                <h2 class="text-base font-bold text-white flex items-center gap-2">
                    <i data-feather="crosshair" class="w-4 h-4 text-purple-400"></i> Recent Bets (Union)
                </h2>
            </div>
            <div class="overflow-y-auto max-h-[500px]">
                <?php if(empty($bets)): ?>
                    <div class="p-8 text-center text-slate-500 font-medium text-sm">No game activity found.</div>
                <?php else: ?>
                    <table class="w-full text-left border-collapse">
                        <thead class="sticky top-0 bg-slate-900 z-10">
                            <tr class="border-b border-slate-700/50">
                                <th class="p-3 pl-5 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Time</th>
                                <th class="p-3 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Game</th>
                                <th class="p-3 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Stake</th>
                                <th class="p-3 pr-5 text-[10px] font-bold text-slate-400 uppercase tracking-widest text-right">Outcome</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-slate-800/50">
                            <?php foreach($bets as $bet): 
                                $won = $bet['win'] > 0;
                            ?>
                            <tr class="hover:bg-slate-800/40 transition-colors">
                                <td class="p-3 pl-5 text-[11px] text-slate-500 font-mono"><?= date('H:i:s M d', strtotime($bet['created_at'])) ?></td>
                                <td class="p-3 font-bold text-slate-300 text-xs tracking-wider uppercase"><?= $bet['game'] ?></td>
                                <td class="p-3 font-medium text-white tracking-tight text-sm">₹<?= money($bet['bet']) ?></td>
                                <td class="p-3 pr-5 text-right font-black tracking-tight text-sm <?= $won ? 'text-green-400' : 'text-slate-500' ?>">
                                    <?= $won ? '+₹'.money($bet['win']) : '-₹'.money($bet['bet']) ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

    </div>
<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
