<?php
require_once 'includes/auth.php';

$msg = '';
$msgType = 'success';

// --- Recursive Salary Core Logic ---
function get_salary_stats(PDO $pdo, string $referrerId) {
    $chk = $pdo->prepare("SELECT user_id, commission_rate FROM users WHERE user_id = ?");
    $chk->execute([$referrerId]);
    $referrer = $chk->fetch(PDO::FETCH_ASSOC);
    if (!$referrer) return null;

    $baseRate = (float)($referrer['commission_rate'] ?? 0);

    $setStmt = $pdo->prepare("SELECT v FROM settings WHERE k = ?");
    $setStmt->execute(['rates_' . $referrerId]);
    $ratesJson = $setStmt->fetchColumn();

    $parsedRates = [];
    if ($ratesJson) {
        $parsed = json_decode($ratesJson, true);
        if (is_array($parsed)) $parsedRates = $parsed;
    }

    $lvlRates = [
        1 => isset($parsedRates[1]) ? (float)$parsedRates[1] : $baseRate,
        2 => isset($parsedRates[2]) ? (float)$parsedRates[2] : 0,
        3 => isset($parsedRates[3]) ? (float)$parsedRates[3] : 0,
        4 => isset($parsedRates[4]) ? (float)$parsedRates[4] : 0,
        5 => isset($parsedRates[5]) ? (float)$parsedRates[5] : 0,
        6 => isset($parsedRates[6]) ? (float)$parsedRates[6] : 0,
    ];

    $todayStmt = $pdo->prepare("SELECT COUNT(user_id) FROM users WHERE referred_by = ? AND DATE(created_at) = CURDATE()");
    $todayStmt->execute([$referrerId]);
    $referredToday = (int)$todayStmt->fetchColumn();

    $levels = [];
    $levels[1] = [];

    $stmt = $pdo->prepare("SELECT user_id FROM users WHERE referred_by = ?");
    $stmt->execute([$referrerId]);
    $levels[1] = $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];

    for ($lvl = 2; $lvl <= 6; $lvl++) {
        $prev = $levels[$lvl - 1];
        if (empty($prev)) { $levels[$lvl] = []; continue; }
        $in = implode(',', array_fill(0, count($prev), '?'));
        $stmt = $pdo->prepare("SELECT user_id FROM users WHERE referred_by IN ($in)");
        $stmt->execute($prev);
        $levels[$lvl] = $stmt->fetchAll(PDO::FETCH_COLUMN);
    }

    $collected = [];
    $userMeta = [];
    $allToQuery = [];

    for ($lvl = 1; $lvl <= 6; $lvl++) {
        foreach ($levels[$lvl] as $uid) {
            if (!in_array($uid, $allToQuery, true)) $allToQuery[] = $uid;
        }
    }

    if (!empty($allToQuery)) {
        $in = implode(',', array_fill(0, count($allToQuery), '?'));
        $map = $pdo->prepare("SELECT user_id, COALESCE(referred_by,'') ref FROM users WHERE user_id IN ($in)");
        $map->execute($allToQuery);
        $refMap = [];
        foreach ($map->fetchAll(PDO::FETCH_ASSOC) as $r) $refMap[$r['user_id']] = $r['ref'];
    } else {
        $refMap = [];
    }

    for ($lvl = 1; $lvl <= 6; $lvl++) {
        foreach ($levels[$lvl] as $uid) {
            if (!isset($userMeta[$uid])) {
                $userMeta[$uid] = ['referrer_id' => $refMap[$uid] ?? '', 'level' => $lvl];
                $collected[] = $uid;
            } elseif ($lvl < $userMeta[$uid]['level']) {
                $userMeta[$uid]['level'] = $lvl;
            }
        }
    }

    $referredCount = count($collected);
    $sumDeposits = $sumWithdraws = $pendingWithdrawals = 0.0;
    $grossCommission = $alreadyPaid = $available = 0.0;
    $list = [];

    if ($referredCount > 0) {
        $inQ = implode(',', array_fill(0, $referredCount, '?'));

        $dep = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM transactions WHERE type='deposit' AND status='completed' AND user_id IN ($inQ)");
        $dep->execute($collected);
        $sumDeposits = (float)$dep->fetchColumn();

        $wd = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM transactions WHERE type='withdrawal' AND status='completed' AND user_id IN ($inQ)");
        $wd->execute($collected);
        $sumWithdraws = (float)$wd->fetchColumn();

        $pending = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM transactions WHERE type='withdrawal' AND status='pending' AND user_id IN ($inQ)");
        $pending->execute($collected);
        $pendingWithdrawals = (float)$pending->fetchColumn();

        $userDep = $pdo->prepare("SELECT user_id, COALESCE(SUM(amount),0) s FROM transactions WHERE type='deposit' AND status='completed' AND user_id IN ($inQ) GROUP BY user_id");
        $userDep->execute($collected);
        $depMap = [];
        foreach ($userDep->fetchAll(PDO::FETCH_ASSOC) as $r) $depMap[$r['user_id']] = (float)$r['s'];

        $userWd = $pdo->prepare("SELECT user_id, COALESCE(SUM(amount),0) s FROM transactions WHERE type='withdrawal' AND status='completed' AND user_id IN ($inQ) GROUP BY user_id");
        $userWd->execute($collected);
        $wdMap = [];
        foreach ($userWd->fetchAll(PDO::FETCH_ASSOC) as $r) $wdMap[$r['user_id']] = (float)$r['s'];

        foreach ($collected as $uid) {
            $rd = $depMap[$uid] ?? 0;
            $rw = $wdMap[$uid] ?? 0;
            $net = $rd - $rw;
            $lvl = $userMeta[$uid]['level'];
            
            $r_applied = $lvlRates[$lvl] ?? 0;
            $userCommission = 0;
            if ($net > 0 && $r_applied > 0) {
                $userCommission = $net * ($r_applied / 100); 
                $grossCommission += $userCommission;
            }

            $list[] = [
                'user_id' => $uid,
                'referrer_id' => $userMeta[$uid]['referrer_id'] ?? '',
                'level' => $lvl,
                'rate_applied' => $r_applied,
                'commission_earned' => round($userCommission, 2),
                'total_deposits' => round($rd,2),
                'total_withdrawals' => round($rw,2),
                'net' => round($net,2),
            ];
        }
    }

    if ($referredCount > 0) {
        $paidStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM referral_payouts WHERE referrer_id = ?");
        $paidStmt->execute([$referrerId]);
        $alreadyPaid = (float)$paidStmt->fetchColumn();

        $available = $grossCommission - $alreadyPaid - $pendingWithdrawals;
    }

    return [
        'user_id' => $referrerId,
        'rate' => $baseRate,
        'rates' => $lvlRates,
        'team_count' => $referredCount,
        'referred_today' => $referredToday,
        'deposits' => $sumDeposits,
        'withdrawals' => $sumWithdraws,
        'pending_withdrawals' => $pendingWithdrawals,
        'net' => $sumDeposits - $sumWithdraws,
        'gross_comm' => $grossCommission,
        'paid_comm' => $alreadyPaid,
        'available' => $available,
        'referred_users' => $list
    ];
}

// Handle Form Posts
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (($_POST['action'] ?? '') === 'pay_salary') {
        $uid = trim($_POST['user_id']);
        
        try {
            $db->beginTransaction();
            
            $chk = $db->prepare("SELECT user_id FROM users WHERE user_id = ? FOR UPDATE");
            $chk->execute([$uid]);
            if (!$chk->fetch()) throw new Exception("User not found: " . h($uid));

            $stats = get_salary_stats($db, $uid);
            $claimAmt = round($stats['available'], 2);

            if ($claimAmt <= 0.01) throw new Exception("No commission available to claim (Amt: ₹$claimAmt)");

            $payStmt = $db->prepare("INSERT INTO referral_payouts (referrer_id, amount, created_at) VALUES (?, ?, NOW())");
            $payStmt->execute([$uid, $claimAmt]);

            $updStmt = $db->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?");
            $updStmt->execute([$claimAmt, $uid]);

            $tid = 'COM' . uniqid();
            $txStmt = $db->prepare("INSERT INTO transactions (transaction_id, user_id, type, amount, status, created_at) VALUES (?, ?, 'referral', ?, 'completed', NOW())");
            $txStmt->execute([$tid, $uid, $claimAmt]);

            $db->commit();
            $msg = "Successfully transferred ₹" . money($claimAmt) . " available commission to " . h($uid) . "'s wallet.";
            $msgType = 'success';
        } catch (Exception $e) {
            if ($db->inTransaction()) $db->rollBack();
            $msg = $e->getMessage();
            $msgType = 'error';
        }
    }
}

$search_user = trim($_GET['search_user'] ?? '');
$stats = null;
if ($search_user) {
    // Only fetch stats if searched
    $stats = get_salary_stats($db, $search_user);
    if (!$stats) {
        $msg = "Agent not found.";
        $msgType = 'error';
    }
}

require_once 'includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-2xl font-bold text-white tracking-tight">Salary Pro</h1>
    <p class="text-sm text-slate-400 mt-1">Multi-Level referral metrics and forced commission payouts.</p>
</div>

<?php if($msg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border <?= $msgType === 'success' ? 'bg-green-500/10 border-green-500/20 text-green-400' : 'bg-red-500/10 border-red-500/20 text-red-400' ?>">
        <i data-feather="<?= $msgType === 'success' ? 'check-circle' : 'alert-circle' ?>" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($msg) ?></p>
    </div>
<?php endif; ?>

<!-- Search Card -->
<div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 p-6 mb-8">
    <form method="GET" class="flex flex-col md:flex-row gap-4">
        <div class="flex-1">
            <label class="text-xs font-bold text-slate-400 uppercase tracking-widest pl-1">Agent / User ID</label>
            <div class="relative mt-2">
                <i data-feather="search" class="w-5 h-5 absolute left-4 top-1/2 -translate-y-1/2 text-slate-500"></i>
                <input type="text" name="search_user" placeholder="e.g. 25111326" value="<?= h($search_user) ?>" class="w-full bg-slate-900 border border-slate-700 rounded-xl pl-12 pr-4 py-3.5 text-white placeholder:text-slate-600 focus:outline-none focus:ring-2 focus:ring-blue-500/50 transition-all font-bold tracking-wider" required>
            </div>
        </div>
        <div class="flex items-end">
            <button type="submit" class="w-full md:w-auto bg-blue-600 hover:bg-blue-500 text-white px-8 py-3.5 rounded-xl font-bold transition-all shadow-lg shadow-blue-500/20">Fetch Network</button>
        </div>
    </form>
</div>

<?php if($stats): ?>
    <!-- Agent Card -->
    <div class="bg-gradient-to-br from-blue-900/40 to-indigo-900/20 border border-blue-500/30 rounded-2xl shadow-xl shadow-blue-900/20 p-6 mb-8 relative overflow-hidden">
        <div class="absolute -right-12 -top-12 w-48 h-48 bg-blue-500/20 rounded-full blur-3xl pointer-events-none"></div>
        
        <div class="flex flex-col lg:flex-row justify-between lg:items-center gap-6 relative z-10">
            <div>
                <div class="flex items-center gap-3 mb-1">
                    <h2 class="text-3xl font-black text-white"><?= h($stats['user_id']) ?></h2>
                    <span class="bg-blue-500/20 text-blue-400 border border-blue-500/30 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">Salary Pro Agent</span>
                </div>
                <p class="text-slate-400 font-medium">Network Size: <span class="text-white font-bold"><?= number_format($stats['team_count']) ?> members</span></p>
                <p class="text-slate-400 font-medium text-sm">Joined Today: <span class="text-white font-bold"><?= number_format($stats['referred_today']) ?> users</span></p>
            </div>

            <div class="flex items-center gap-2">
                <div class="bg-slate-900/80 border border-slate-700/50 rounded-xl p-3 flex flex-col gap-1.5 min-w-[140px]">
                    <span class="text-[10px] font-bold text-slate-500 uppercase tracking-widest text-center">Levels 1-3</span>
                    <div class="flex justify-between text-xs font-bold font-mono">
                        <span class="text-indigo-400">L1: <?= $stats['rates'][1] ?>%</span>
                        <span class="text-indigo-400">L2: <?= $stats['rates'][2] ?>%</span>
                        <span class="text-indigo-400">L3: <?= $stats['rates'][3] ?>%</span>
                    </div>
                </div>
                <div class="bg-slate-900/80 border border-slate-700/50 rounded-xl p-3 flex flex-col gap-1.5 min-w-[140px]">
                    <span class="text-[10px] font-bold text-slate-500 uppercase tracking-widest text-center">Levels 4-6</span>
                    <div class="flex justify-between text-xs font-bold font-mono">
                        <span class="text-indigo-400">L4: <?= $stats['rates'][4] ?>%</span>
                        <span class="text-indigo-400">L5: <?= $stats['rates'][5] ?>%</span>
                        <span class="text-indigo-400">L6: <?= $stats['rates'][6] ?>%</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="grid grid-cols-2 lg:grid-cols-5 gap-4 mt-8 pt-8 border-t border-slate-700/50 relative z-10">
            <div>
                <p class="text-xs font-bold text-slate-500 uppercase tracking-widest mb-1">Network Deposits</p>
                <p class="text-xl font-black text-white">₹<?= money($stats['deposits']) ?></p>
            </div>
            <div>
                <p class="text-xs font-bold text-slate-500 uppercase tracking-widest mb-1">Network Withdrawn</p>
                <p class="text-xl font-black text-white">₹<?= money($stats['withdrawals']) ?></p>
            </div>
            <div>
                <p class="text-xs font-bold text-slate-500 uppercase tracking-widest mb-1">Gross Commission</p>
                <p class="text-xl font-black text-white">₹<?= money($stats['gross_comm']) ?></p>
            </div>
            <div>
                <p class="text-xs font-bold text-slate-500 uppercase tracking-widest mb-1">Already Paid</p>
                <p class="text-xl font-black text-orange-400">₹<?= money($stats['paid_comm']) ?></p>
            </div>
            <div class="bg-green-500/10 border border-green-500/20 rounded-xl p-3 -mt-3">
                <p class="text-[10px] font-bold text-green-500 uppercase tracking-widest mb-1">Available to Claim</p>
                <p class="text-2xl font-black text-green-400">₹<?= money($stats['available']) ?></p>
            </div>
        </div>

        <?php if($stats['available'] > 0.01): ?>
            <form method="POST" class="mt-6 flex justify-end relative z-10" onsubmit="return confirm('Transfer ₹<?= money($stats['available']) ?> to <?= h($stats['user_id']) ?> immediately?');">
                <input type="hidden" name="action" value="pay_salary">
                <input type="hidden" name="user_id" value="<?= $stats['user_id'] ?>">
                <button type="submit" class="bg-green-600 hover:bg-green-500 text-white font-bold py-3 px-8 rounded-xl shadow-lg shadow-green-600/20 transition-all flex items-center gap-2">
                    <i data-feather="corner-down-right" class="w-5 h-5"></i> Execute Payout
                </button>
            </form>
        <?php endif; ?>
    </div>

    <!-- Network Drilldown -->
    <div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden">
        <div class="p-6 border-b border-slate-700/50">
            <h3 class="text-lg font-bold text-white flex items-center gap-2"><i data-feather="git-merge" class="w-5 h-5 text-indigo-400"></i> Network Drilldown</h3>
        </div>
        
        <?php if(empty($stats['referred_users'])): ?>
            <div class="p-8 text-center text-slate-500 font-medium">This user has not referred anyone yet.</div>
        <?php else: ?>
            <div class="overflow-x-auto">
                <table class="w-full text-left border-collapse">
                    <thead>
                        <tr class="bg-slate-900/50 border-b border-slate-700/50">
                            <th class="p-4 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Downline User</th>
                            <th class="p-4 text-[10px] font-bold text-slate-400 uppercase tracking-widest">Hierarchy Depth</th>
                            <th class="p-4 text-[10px] font-bold text-slate-400 uppercase tracking-widest right-align">Net Loss (Rev)</th>
                            <th class="p-4 text-[10px] font-bold text-slate-400 uppercase tracking-widest right-align">Comm Amt</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-slate-800/50">
                        <?php foreach($stats['referred_users'] as $ru): ?>
                        <tr class="hover:bg-slate-800/40 transition-colors">
                            <td class="p-4 font-bold text-slate-200">
                                <?= h($ru['user_id']) ?>
                                <?php if($ru['referrer_id'] !== $stats['user_id']): ?>
                                    <div class="text-[10px] text-slate-500 font-normal mt-0.5">via <?= h($ru['referrer_id']) ?></div>
                                <?php endif; ?>
                            </td>
                            <td class="p-4">
                                <span class="bg-slate-700 text-slate-300 border border-slate-600 px-2 py-0.5 rounded text-[10px] font-bold tracking-wider">Lvl <?= $ru['level'] ?></span>
                                <span class="text-[10px] text-slate-500 ml-2">Rate: <?= $ru['rate_applied'] ?>%</span>
                            </td>
                            <td class="p-4 text-right font-mono text-sm <?= $ru['net'] > 0 ? 'text-green-400' : 'text-slate-500' ?>">
                                <?= $ru['net'] > 0 ? '+₹' . money($ru['net']) : '₹' . money($ru['net']) ?>
                            </td>
                            <td class="p-4 text-right font-bold text-indigo-400">
                                ₹<?= money($ru['commission_earned']) ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
